spool storet_wme_pkg.log

PROMPT Creating Package Body 'STORET_WME_PKG'

SET DEF OFF

CREATE OR REPLACE PACKAGE BODY STORET_WME_PKG IS
/*********************************************************
Filename			STORET_WME_PKG
Release			E2.0
SIR			1552 and 1553
Description		STORET Station Home Page that provides summary station information and a bar chart that
			depicts the number of results per reported characteristic.  Page is called from the WME map page and linked
			when a STORET station is picked from a list of retrieved stations.
Author			Sue Kho/Blythe Norris
Date			08/06/2003
Comments		09/25/2003 --Task 26 begins.  Modified to incorporate appropriate links to the Results by Station page (Central DWH).  
			DW "model change" (i.e., ETL script change) required to add count types in the Station_Char table.  Add page to give 
			these counts if more than one is populated.  B. Norris

			09/29/2003 - The URL that opens the STORET Station Home Page from the WME map page pads the Station ID parameter 
			with "%20".   This can be fixed by updating the data type of the view for that field to VARCHAR2.  Lee Manning said to leave 
			this as is for now.  B. Norris
			11/19/2003 - Add URL input parameter error trapping logic to Display_Station procedure.  B Norris
			11/24/2003 - Add second view and logic to display station summary information for stations not yet visited.  B. Norris
*********************************************************/             
PROCEDURE initXmlStream(
	xl IN NUMBER,
	yt IN NUMBER,
	xr IN NUMBER,
	 yb IN NUMBER)  
IS
BEGIN
                  dbxml_wme.Query('SELECT "PkISNIdentifier", "StationIdentifier", "StationName", "OrganizationIdentifier", ' 
		|| ' "OrganizationFormalName", "PrimaryTypeName", "LatitudeMeasure", "LongitudeMeasure", ' 
		|| ' "HorizontalReferenceDatumName", "HorizontalCollectionMethodText", "CountryName",' 
		|| ' "HydrologicUnitCode", "HydrologicUnitName", "NativeAmericanLandName","StationSGOIndicator" '
		|| ' FROM STORET_STATION_WME_VW'
		|| ' WHERE "LatitudeMeasure" <= ' || TO_CHAR(yt)
		|| ' AND "LatitudeMeasure"  >= ' || TO_CHAR(yb)
		|| ' AND "LongitudeMeasure" <= ' || TO_CHAR(xr)
		|| ' AND "LongitudeMeasure" >= ' || TO_CHAR(xl)
		|| ' order by "OrganizationIdentifier" asc,  "StationIdentifier" asc' );	
END;


/*********************************************************/             
FUNCTION add_comma (p_text IN VARCHAR2) 
	RETURN VARCHAR2
IS
BEGIN
	IF (NVL(LENGTH(p_text), 0) != 0) THEN
		RETURN ( ', '||   '&' || 'nbsp;' || '&' || 'nbsp;' || '&' || 'nbsp;'  || p_text);
	END IF;
	RETURN (p_text);
END;


/*********************************************************/             
PROCEDURE Display_Station_List
IS
	 CURSOR curr_station IS
		SELECT "PkISNIdentifier", "StationIdentifier", "StationName", "OrganizationIdentifier"
			FROM STORET_STATION_WME_VW
			WHERE "OrganizationIdentifier" = 'UTAHDWQ' OR "OrganizationIdentifier" = 'DEMODWH'
			ORDER BY "StationIdentifier";
	Double_Quote VARCHAR2(2) := '"';
BEGIN

--The procedure is only used for internal testing when WME map page was down/unavailable.

	WME_header('STORET Station List', 'Station List');
	htp.centerOpen;
	htp.p('<FORM ACTION="Display_Station"  METHOD = "POST" NAME = "form1">');
	htp.bodyOpen;

           	htp.tableOpen('BORDER=1');
	htp.p(htf.bold('<TH>Station Identifier</TH><TH>Station Name</TH>'));
	FOR rec_station IN curr_station LOOP
     		htp.tableRowOpen;
         		htp.tableData('<A HREF=' ||  Double_Quote || g_App_Path || 'storet_wme_pkg.Display_Station?p_station_id=' 
			|| RTRIM(rec_station."StationIdentifier") || '&' || 'p_org_id=' || rec_station."OrganizationIdentifier" 
			|| Double_Quote || '>' || rec_station."StationIdentifier" ,  'Left');
         		htp.tableData(rec_station."StationName" || '<BR>', 'Left');
     		htp.tableRowClose;
     	END LOOP;
	
     	htp.tableClose;
	htp.bodyClose;

	WME_footer();
END;


/*********************************************************/             
/* *** STORET Station Home page ***/
PROCEDURE Display_Station (
	p_station_id IN VARCHAR2 DEFAULT NULL,
	p_org_id   IN VARCHAR2 DEFAULT NULL)
IS
	p_pk_isn_id NUMBER(12) := NULL;
	message_id VARCHAR2(1000) := NULL;
	p_station_cnt NUMBER := 0;
	p_org_cnt NUMBER := 0;
	p_visited CHAR(1) := 'Y';

BEGIN
	WME_header('STORET Station Home Page', 'Station Information');

--Add URL input parameter error trapping logic for cases in which this page is opened without a front end (e.g., WME)

--Null or blank parameter check

	IF p_station_id IS NULL THEN
  		message_id := 'The Station ID input parameter p_station_id is missing or has been left blank.';
	ELSIF  p_org_id IS NULL THEN
	 	message_id := 'The Organization ID input parameter p_org_id is missing or has been left blank.';
	ELSE

--Invalid Org ID check

		SELECT count(pk_isn)  INTO p_org_cnt
			FROM storetw.DI_ORG
			WHERE 	rtrim( ORGANIZATION_ID) = rtrim(p_org_id);

		IF p_org_cnt = 0 THEN
			message_id := 'Organization ID '||p_org_id||' not found in STORET.';
		END IF;

	END IF;

--Valid, visited Station

	IF message_id IS NULL THEN

		SELECT count("PkISNIdentifier")  INTO p_station_cnt
			FROM STORET_STATION_WME_VW
			WHERE 	rtrim("StationIdentifier") = rtrim(p_station_id) AND
				rtrim("OrganizationIdentifier") = rtrim(p_org_id);					

		IF p_station_cnt > 0 THEN
			SELECT "PkISNIdentifier"  INTO p_pk_isn_id
				FROM STORET_STATION_WME_VW
				WHERE 	rtrim( "StationIdentifier") = rtrim(p_station_id) AND
					rtrim("OrganizationIdentifier") = rtrim(p_org_id);	

			getDetail(p_station_id, p_pk_isn_id,p_visited);

			htp.br;
			htp.br;
	
			Station_Results(p_station_id, p_pk_isn_id);

		ELSE

--Valid, not visited Station

			SELECT count("PkISNIdentifier")  INTO p_station_cnt
				FROM STORET_STATION_NV_VW
				WHERE 	rtrim("StationIdentifier") = rtrim(p_station_id) AND
					rtrim("OrganizationIdentifier") = rtrim(p_org_id);	

			IF p_station_cnt > 0 THEN

				message_id := 'No monitoring data to display for Organization ID '||p_org_id||', Station ID '||p_station_id||'.';

--Show Station Summary information only
				SELECT "PkISNIdentifier"  INTO p_pk_isn_id
				FROM STORET_STATION_NV_VW
				WHERE 	rtrim( "StationIdentifier") = rtrim(p_station_id) AND
					rtrim("OrganizationIdentifier") = rtrim(p_org_id);	

				p_visited := 'N';
				getDetail(p_station_id, p_pk_isn_id,p_visited);		
			
			ELSE
	
--Invalid Station ID
				message_id := 'Organization ID '||p_org_id||', Station ID '||p_station_id||' not found in STORET. ';	

			END IF;
		END IF;
	END IF;
	IF message_id is not NULL THEN
		htp.p('<br><br><center>'||message_id||'<br/>');
		htp.br;
		htp.br;
	END IF;

	htp.p('<br><br><center>Comments? If you have a question or comments on this website or on the STORET system,<br />');	
	htp.p('please call STORET User Assistance at 1-800-424-9067,<br />');
	htp.p('or send us an email at <a href="mailto:STORET@epa.gov" target="_top">STORET@epa.gov</a>.</center>');

	WME_footer();

EXCEPTION
	WHEN OTHERS THEN

		htp.br;
		htp.br;
		htp.p('<br><br><center>'||'Input parameters are invalid.'||'<br/>');
		htp.br;
		htp.br;

		htp.p('<br><br><center>Comments? If you have a question or comments on this website or on the STORET system,<br/>');	
		htp.p('please call STORET User Assistance at 1-800-424-9067,<br />');
		htp.p('or send us an email at <a href="mailto:STORET@epa.gov" target="_top">STORET@epa.gov</a>.</center>');

		WME_footer();

END;


/*********************************************************/             
/* *** Display station detail information ***/
-- **** Possible to include:  DatabaseRefreshDate
PROCEDURE GetDetail(
	p_station_id IN VARCHAR2,
	p_pk_isn_id IN NUMBER,
	p_visited IN CHAR)
 IS
   	CURSOR curr_station (p_stat_id char) IS
		SELECT * FROM STORET_STATION_WME_VW
		WHERE "StationIdentifier" = p_stat_id AND  "PkISNIdentifier" =p_pk_isn_id ;

  	CURSOR curr_station_nv (p_stat_id char) IS
		SELECT * FROM STORET_STATION_NV_VW
		WHERE "StationIdentifier" = p_stat_id AND  "PkISNIdentifier" =p_pk_isn_id ;

	spaces VARCHAR2(20) := '&' || 'nbsp;' || '&' || 'nbsp;' || '&' || 'nbsp;' ;
	l_firsttime BOOLEAN := TRUE;
BEGIN
	htp.centerOpen;

	IF p_visited = 'Y' THEN

--Visited Station

	    FOR rec_station IN curr_station(p_station_id) LOOP
		IF l_firsttime THEN
			htp.tableOpen;
			htp.tableRowOpen;
			htp.p('<TD>');
			     	htp.tableOpen;
				htp.tableRowOpen;
					htp.tableData(htf.bold('Station: '),  'Left');
		         			htp.tableData(rec_station."StationIdentifier" ,  'Left');
		         			htp.tableData(spaces || rec_station."StationName",  'Left');
				htp.tableRowClose;
				htp.tableRowOpen;
					htp.tableData(htf.bold('Organization: '),  'Left');
		         			htp.tableData(rec_station."OrganizationIdentifier",   'Left');
		         			htp.tableData(spaces || rec_station."OrganizationFormalName" ,  'Left');
				htp.tableRowClose;
				htp.tableClose;
			htp.p('</TD><TD VALIGN=TOP><IMG ALT="STORET logo" SRC="' || g_Gif_Path || 'storet_logo.gif"></TD>');
			htp.tableRowClose;
			htp.tableClose;
			l_firsttime  := FALSE;
		END IF;

--Make border as narrow as possible.  Using <table> instead of tableOpen gives you cellspacing and cellpadding that helps minimize blank space.

		htp.p('<table border="1" cellspacing="0" cellpadding="0">');
--           		htp.tableOpen('BORDER=1');
     		htp.tableRowOpen;
		htp.p('<TD ALIGN=Center COLSPAN=2><B>Station Information</B></TD>');
        		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Primary Type'),  'Left');
         			htp.tableData(rec_station."PrimaryTypeName" || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Secondary Type'),  'Left');
         			htp.tableData(rec_station."SecondaryTypeName" || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Latitude/Longtitude'),  'Left');
			htp.tableData('Latitude ' || rec_station."LatitudeMeasure" || ', '  || spaces
				||  'Longitude ' || rec_station."LongitudeMeasure" 
				|| add_comma(rec_station."HorizontalReferenceDatumCode"),  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Elevation'),  'Left');

--SIR 1569, TIR 0001 Do not show elevation if equal to "0" (without units).  This is not possible in the data entry application.

			IF RTRIM(rec_station."VerticalMeasure")  IS NULL OR RTRIM(rec_station."VerticalMeasure") = '0' THEN
				htp.tableData('<BR>');
			ELSE	
				htp.tableData(RTRIM(rec_station."VerticalMeasure") || add_comma(rec_station."VerticalReferenceDatumCode") || '<BR>',  'Left');
			END IF;
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('State'),  'Left');
			htp.tableData(rec_station."StateName" || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
              			htp.tableData(htf.bold('County'),  'Left');
      			htp.tableData(rec_station."CountyName"  || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
		htp.tableData(htf.bold('Hydrologic Unit'),  'Left');
		htp.tableData(rec_station."HydrologicUnitCode" || add_comma(rec_station."HydrologicUnitName") || '<BR>',  'Left');
         		htp.tableRowClose;
		IF (rec_station."OceanName" IS NOT NULL OR rec_station."PrimaryEstuaryName" IS NOT NULL OR rec_station."GreatLakeName" IS NOT NULL) THEN
	         		htp.tableRowOpen;
			IF rec_station."OceanName" IS NOT NULL THEN
				htp.tableData(htf.bold('Ocean'),  'Left');
				htp.tableData(rec_station."OceanName",  'Left');
			ELSIF rec_station."PrimaryEstuaryName"  IS NOT NULL OR rec_station."SecondaryEstuaryName"  IS NOT NULL THEN
				htp.tableData(htf.bold('10/20 Estuary'),  'Left');
				htp.tableData(rec_station."PrimaryEstuaryName" || ' /  ' || rec_station."SecondaryEstuaryName" ,  'Left');
			ELSIF rec_station."GreatLakeName" IS NOT NULL THEN 
				htp.tableData(htf.bold('Great Lake'),  'Left');
				htp.tableData(rec_station."GreatLakeName",  'Left');
			END IF;
        			htp.tableRowClose;
		END IF;
		
--         		htp.tableClose;
		htp.p('</table>');
     	    END LOOP;

	ELSE

--Non-visited Station

    		FOR rec_station IN curr_station_nv(p_station_id) LOOP
		IF l_firsttime THEN
			htp.tableOpen;
			htp.tableRowOpen;
			htp.p('<TD>');
			     	htp.tableOpen;
				htp.tableRowOpen;
					htp.tableData(htf.bold('Station: '),  'Left');
		         			htp.tableData(rec_station."StationIdentifier" ,  'Left');
		         			htp.tableData(spaces || rec_station."StationName",  'Left');
				htp.tableRowClose;
				htp.tableRowOpen;
					htp.tableData(htf.bold('Organization: '),  'Left');
		         			htp.tableData(rec_station."OrganizationIdentifier",   'Left');
		         			htp.tableData(spaces || rec_station."OrganizationFormalName" ,  'Left');
				htp.tableRowClose;
				htp.tableClose;
			htp.p('</TD><TD VALIGN=TOP><IMG ALT="STORET logo" SRC="' || g_Gif_Path || 'storet_logo.gif"></TD>');
			htp.tableRowClose;
			htp.tableClose;
			l_firsttime  := FALSE;
		END IF;

--Make border as narrow as possible.  Using <table> instead of tableOpen gives you cellspacing and cellpadding that helps minimize blank space.

		htp.p('<table border="1" cellspacing="0" cellpadding="0">');
--           		htp.tableOpen('BORDER=1');
     		htp.tableRowOpen;
		htp.p('<TD ALIGN=Center COLSPAN=2><B>Station Information</B></TD>');
        		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Primary Type'),  'Left');
         			htp.tableData(rec_station."PrimaryTypeName" || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Secondary Type'),  'Left');
         			htp.tableData(rec_station."SecondaryTypeName" || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Latitude/Longtitude'),  'Left');
			htp.tableData('Latitude ' || rec_station."LatitudeMeasure" || ', '  || spaces
				||  'Longitude ' || rec_station."LongitudeMeasure" 
				|| add_comma(rec_station."HorizontalReferenceDatumCode"),  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('Elevation'),  'Left');

--SIR 1569, TIR 0001 Do not show elevation if equal to "0" (without units).  This is not possible in the data entry application.

			IF RTRIM(rec_station."VerticalMeasure")  IS NULL OR RTRIM(rec_station."VerticalMeasure") = '0' THEN
				htp.tableData('<BR>');
			ELSE	
				htp.tableData(RTRIM(rec_station."VerticalMeasure") || add_comma(rec_station."VerticalReferenceDatumCode") || '<BR>',  'Left');
			END IF;
         		htp.tableRowClose;
	         	htp.tableRowOpen;
           			htp.tableData(htf.bold('State'),  'Left');
			htp.tableData(rec_station."StateName" || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
              			htp.tableData(htf.bold('County'),  'Left');
      			htp.tableData(rec_station."CountyName"  || '<BR>',  'Left');
         		htp.tableRowClose;
	         	htp.tableRowOpen;
		htp.tableData(htf.bold('Hydrologic Unit'),  'Left');
		htp.tableData(rec_station."HydrologicUnitCode" || add_comma(rec_station."HydrologicUnitName") || '<BR>',  'Left');
         		htp.tableRowClose;
		IF (rec_station."OceanName" IS NOT NULL OR rec_station."PrimaryEstuaryName" IS NOT NULL OR rec_station."GreatLakeName" IS NOT NULL) THEN
	         		htp.tableRowOpen;
			IF rec_station."OceanName" IS NOT NULL THEN
				htp.tableData(htf.bold('Ocean'),  'Left');
				htp.tableData(rec_station."OceanName",  'Left');
			ELSIF rec_station."PrimaryEstuaryName"  IS NOT NULL OR rec_station."SecondaryEstuaryName"  IS NOT NULL THEN
				htp.tableData(htf.bold('10/20 Estuary'),  'Left');
				htp.tableData(rec_station."PrimaryEstuaryName" || ' /  ' || rec_station."SecondaryEstuaryName" ,  'Left');
			ELSIF rec_station."GreatLakeName" IS NOT NULL THEN 
				htp.tableData(htf.bold('Great Lake'),  'Left');
				htp.tableData(rec_station."GreatLakeName",  'Left');
			END IF;
        			htp.tableRowClose;
		END IF;
		
--         		htp.tableClose;
		htp.p('</table>');
     	    END LOOP;

	END IF;
	htp.centerClose;
END;


/*********************************************************/                             
/* Calculate the relative width of the displayed bar */
FUNCTION Calculate_Width (
	p_count IN NUMBER,
	p_TempWidth IN NUMBER DEFAULT 0,
	p_IsGreater IN BOOLEAN DEFAULT FALSE)
		RETURN NUMBER
IS
BEGIN
	IF p_IsGreater THEN	RETURN( p_count / p_TempWidth);
	ELSE 			RETURN( p_count * p_TempWidth);
	END IF;
	RETURN(0);
END;

/*********************************************************/       
/*    Display number of results count by characteristic with horizontal bar graph */
/*    Add link to associated "Results by Station" page on the Characteristic name  */
 PROCEDURE Station_Results (
	p_station_id IN VARCHAR2,
	p_pk_isn_id IN NUMBER)
 IS

--  Add three new counts (09/30/2003).  B. Norris

	CURSOR curr_station IS
		SELECT display_name, result_count, regular_count,habitat_count,biological_count,fk_char
			FROM storetw.STATION_CHAR
			WHERE fk_fa_statn = p_pk_isn_id
			ORDER BY result_count DESC,  display_name;


	l_firsttime BOOLEAN DEFAULT TRUE;
	l_IsGreater BOOLEAN  DEFAULT FALSE;
	l_TempWidth NUMBER(7,2) := 0;
	l_result_url VARCHAR2(200) DEFAULT NULL;

BEGIN
	htp.centerOpen;

	g_MaxCount := 0;
	SELECT MAX (RESULT_COUNT)  INTO g_MaxCount
		FROM storetw.STATION_CHAR  WHERE fk_fa_statn = p_pk_isn_id;

	IF g_MaxCount IS NULL THEN
		htp.p(htf.bold('No Results were found for this Station.'));
	ELSE
		IF g_MaxCount > g_MaxWidth THEN
			l_TempWidth := g_MaxCount / g_MaxWidth;
			l_IsGreater := TRUE;
		ELSIF  g_MaxCount <= g_MaxWidth THEN
			l_TempWidth := g_MaxWidth / g_MaxCount;
		END IF;

		FOR rec_station IN curr_station LOOP
			IF l_firsttime THEN

--Make border as narrow as possible.  Using <table> instead of tableOpen gives you cellspacing and cellpadding that helps minimize blank space.

				htp.p('<table width="95%" border="1" cellspacing="0" cellpadding="0">');
--				htp.tableOpen('BORDER=1', NULL, NULL, NULL, 'WIDTH=95%');
	     			htp.tableRowOpen;
				htp.p('<TD ALIGN=Center COLSPAN=2><B>Number of Results</B></TD>');
	        			htp.tableRowClose;
				l_firsttime := FALSE;
			END IF;

-- Determine which "Results by Station" page to go to.

			IF (rec_station.regular_count > 0 AND rec_station.biological_count IS NuLL and rec_station.habitat_count IS NULL) THEN
				l_result_url := 'DW_resultcriteria_station_wme?station_isn=';				
			ELSIF (rec_station.regular_count IS NULL AND rec_station.biological_count > 0 and rec_station.habitat_count IS NULL) THEN
				l_result_url := 'DW_bio_result_station_wme?station_isn=';
			ELSIF (rec_station.regular_count IS NULL AND rec_station.biological_count IS NULL and rec_station.habitat_count > 0)  THEN
				l_result_url := 'DW_hab_result_station_wme?station_isn=';
			ELSE
				l_result_url:='storet_wme_pkg.Char_Detail?station_isn=';
			END IF;				

	      		htp.tableRowOpen;
         				htp.tableData('<A HREF="' || g_App_Path || l_result_url || p_pk_isn_id || '&' ||'station_id='||RTRIM(p_station_id)
					|| '&' ||  'char_isn=' || rec_station.fk_char|| '">' 
					|| rec_station.display_name, 'Left',  NULL, NULL,  NULL,NULL, 'WIDTH=35%' );
--        				htp.tableData('<IMG SRC = c:\temp\steve3.gif  ALIGN=LEFT  ALT="color.gif" border=0 height=13 width=' 
--					|| Calculate_Width(rec_station.result_count, l_TempWidth,  l_IsGreater) 
--					|| '>&' || 'nbsp;'  || rec_station.result_count,  'Left', NULL,  'NOWRAP',  NULL,NULL, 'WIDTH=60%');
	        			htp.tableData('<IMG SRC =' ||  g_Gif_Path || 'green.gif  ALIGN=LEFT  ALT="color.gif" border=0 height=13 width=' 
					|| Calculate_Width(rec_station.result_count, l_TempWidth,  l_IsGreater) 
					|| '>&' || 'nbsp;'  || rec_station.result_count,  'Left', NULL,  'NOWRAP',  NULL,NULL, 'WIDTH=60%');
       			htp.tableRowClose;
	     	END LOOP;
		IF NOT  l_firsttime THEN
--		         	htp.tableClose;
			htp.p('</table>');
		END IF;

	END IF;

	htp.centerClose;
END;

/*********************************************************/    
/* *** Called from Station_Results when there are multiple Result Types to display for a given characteristic ***/
PROCEDURE Char_Detail (	
	station_id IN VARCHAR2,
	station_isn IN NUMBER,
	char_isn IN NUMBER)
 IS
	l_station_name VARCHAR2(255) DEFAULT NULL;
	l_org_id VARCHAR2(8) DEFAULT NULL;
	l_org_name VARCHAR2(255) DEFAULT NULL;
	l_display_name VARCHAR2(60) DEFAULT NULL;
	l_hab_count NUMBER:=0;
	l_bio_count NUMBER:=0;
	l_reg_count NUMBER:=0;

BEGIN
	WME_header('Characteristic Detail Information', 'Characteristic');


	SELECT STATION_NAME, ORGANIZATION_ID INTO l_station_name, l_org_id  FROM storetw.FA_STATION
		WHERE 	PK_ISN = station_isn;

	SELECT ORGANIZATION_NAME INTO l_org_name FROM storetw.DI_ORG
		WHERE ORGANIZATION_ID = l_org_id;

	SELECT DISPLAY_NAME, REGULAR_COUNT,HABITAT_COUNT,BIOLOGICAL_COUNT 
		INTO l_display_name, l_reg_count,l_hab_count,l_bio_count FROM storetw.STATION_CHAR
		WHERE fk_fa_statn = station_isn AND FK_CHAR = char_isn;
	
	htp.centerOpen;
    	htp.tableOpen;
	htp.tableRowOpen;
	htp.p('<TD>');
		htp.tableOpen;
		htp.tableRowOpen;
	          		htp.tableData(htf.bold('Station:'),  'Left');
		   	htp.tableData(station_id,  'Left');
			htp.tableData(l_station_name, 'Left');
		htp.tableRowClose;

		htp.tableRowOpen;
			htp.tableData(htf.bold('Organization: '),  'Left');
			htp.tableData(l_org_id, 'Left');
			htp.tableData(l_org_name, 'Left');
		htp.tableRowClose;
	
		htp.tableRowOpen;
          			htp.tableData(htf.bold('Characteristic:'),  'Left');
		   	htp.tableData(l_display_name,  'Left');
		htp.tableRowClose;
		htp.tableClose;
	htp.p('</TD><TD VALIGN=TOP><IMG ALT="STORET logo" SRC="' || g_Gif_Path || 'storet_logo.gif"></TD>');
	htp.tableRowClose;
	htp.tableClose;	
	htp.br;
	htp.br;

	htp.p(htf.bold('Multiple Result types were found for this Characteristic.  Select the type for which you wish to view Result details.'));
	htp.br;
	htp.br;
	htp.br;
	htp.br;

	htp.p('<table border="1" cellspacing="0" cellpadding="0">');
--	htp.tableOpen('BORDER=0');
	htp.p(htf.bold('<TH>Result Type</TH><TH>Count</TH>'));
	IF (l_reg_count > 0) THEN
		htp.tableRowOpen;
			htp.tableData('<A HREF="' || g_App_Path || 'DW_resultcriteria_station_wme?station_isn='||station_isn||'&' ||'station_id='||RTRIM(station_id)
					|| '&' || 'char_isn='|| char_isn ||'">'|| 'Regular Results','Left' );
			htp.tableData( l_reg_count,'Left');
		htp.tableRowClose;
	END IF;
	IF (l_bio_count > 0) THEN
		htp.tableRowOpen;
			htp.tableData('<A HREF="' || g_App_Path || 'DW_bio_result_station_wme?station_isn='||station_isn||'&' ||'station_id='||RTRIM(station_id)
					|| '&' || 'char_isn='|| char_isn ||'">'|| 'Biological Results','Left' );
			htp.tableData(l_bio_count,'Left');
		htp.tableRowClose;
	END IF;
	IF (l_hab_count > 0) THEN
		htp.tableRowOpen;
			htp.tableData('<A HREF="' || g_App_Path || 'DW_hab_result_station_wme?station_isn='||station_isn||'&' ||'station_id='||RTRIM(station_id)
					|| '&' || 'char_isn='|| char_isn ||'">'|| 'Habitat Results','Left' );
			htp.tableData(l_hab_count, 'Left');
		htp.tableRowClose;
	END IF;
--	htp.tableClose;
	htp.p('</table>');
	htp.centerClose;
	WME_footer();
END;


/*********************************************************/    
/* *** STORET Station Web header *** */
PROCEDURE WME_header(
	page_title 	IN varchar2, 
	p_breadcrumbs      IN VARCHAR2 DEFAULT NULL,
	body_on_load 	IN varchar2 DEFAULT NULL)
IS
	l_Script_Path VARCHAR2(255)  :=DW_glob_var.lv_script_path;
BEGIN
	htp.print('
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN"
	"http://www.w3.org/TR/1999/REC-html401-19991224/loose.dtd">
<html><!-- #BeginTemplate "/Templates/storet.dwt" -->
<!-- EPA Template Version 2.1.1 - 03/22/02 -->
<head>
<!-- #BeginEditable "doctitle" --> 
<title>EPA &gt; STORET &gt; '||page_title||'</title>
<meta name="Description" content="Environmental Protection Agency''s Monitoring and Assessing Water Quality website." />
<meta name="Keywords" content="" />
<SCRIPT LANGUAGE="JavaScript1.2" SRC='|| l_Script_Path||'>
</SCRIPT>
<!-- #EndEditable --> 
<meta http-equiv="Content-Type" content="text/html;  charset=iso-8859-1" />
<meta http-equiv="pics-label" content=''(pics-1.1 "http://www.icra.org/ratingsv02.html" l gen true for "http://www.epa.gov" r (cz 1 lz 1 nz 1 oz 1 vz 1) "http://www.rsac.org/ratingsv01.html" l gen true for "http://www.epa.gov" r (n 0 s 0 v 0 l 0))'' />
<link rel="stylesheet" href="http://www.epa.gov/owow/styles/epafiles_epastyles.css" type="text/css" />
</head>
<body bgcolor="#FFFFFF" text="#000000" 
leftmargin="2" topmargin="0" marginwidth="2" marginheight="0" 
onload='||body_on_load||'>
<a name="pagetop"></a> 
<table width="95%" border="0" cellspacing="0" cellpadding="0">
	<tr valign="top"> 
		<td><a href="#pagecontents"><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
		width="4" height="4" border="0" hspace="68" alt="Skip common site navigation and headers" /></a><br />
			<!-- START EPA SEAL -->
			<a href="http://www.epa.gov/"><img src="http://www.epa.gov/epafiles/images/epafiles_logo_epasealwhite.gif" 
			width="140" height="111" alt="United States Environmental Protection Agency" 
			border="0" /></a> 
			<!-- END EPA SEAL -->
		</td>
		<td width="100%"> 
			<table width="100%" border="0" cellspacing="0" cellpadding="0">
				<tr> 
					<!-- START EPA NAME -->
					<td colspan="3" align="right"><a href="http://www.epa.gov/"><img src="http://www.epa.gov/epafiles/images/epafiles_ban_topepa.gif" 
					width="377" height="26" alt="United States Environmental Protection Agency" border="0" /></a></td>
					<!-- END EPA NAME -->
				</tr>
				<tr> 
					<td class="epaAreaName" rowspan="2"><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
					width="5" height="5" alt="" /></td>
					<td rowspan="2" class="epaAreaName" width="100%"> 
						<!-- START AREA NAME -->
						STORET 
						<!-- END AREA NAME -->
					</td>
					<td valign="top" class="epaAreaName" align="right"><img src="http://www.epa.gov/epafiles/images/epafiles_ban_endtop.gif" 
					width="11" height="14" alt="" /></td>
				</tr>
				<tr> 
					<td valign="bottom" class="epaAreaName" align="right"><img src="http://www.epa.gov/epafiles/images/epafiles_ban_endbottom.gif" 
					width="11" height="15" alt="" /></td>
				</tr>
			</table>
			<table width="100%" border="0" cellspacing="0" cellpadding="0">
				<tr> 
					<td><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
					width="5" height="5" alt="" /></td>
					<td width="100%"> 
						<!-- BEGIN GLOBAL LINKS AND SEARCH -->
						<form name="fcol" method="post" action="http://search.epa.gov/s97is.vts" class="epaGlobal">
							<table width="100%" border="0" cellpadding="0" cellspacing="0">
								<tr> 
									<td><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
									width="4" height="1" vspace="9" alt="" align="middle" /></td>
									<td width="100%" class="epaGlobalLinks"> 
										<!-- START SEARCH CONTROLS -->
										<!-- See http://www.epa.gov/epafiles/searchcode.htm
										for instructions on coding the search box -->
										<input type="hidden" name="fld" value="???" />
										<input type="hidden" name="areaname" value="STORET" />
										<input type="hidden" name="areasidebar" value="http://www.epa.gov/storet/styles/storet_sidebar.txt" />
										<input type="hidden" name="areacontacts" value="http://www.epa.gov/storet/contact.html" />
										<input type="hidden" name="areasearchurl" value="" />
										<input type="hidden" name="typeofsearch" value="" />
										<!-- THE REMAINING HIDDEN INPUTS ARE USED BY THE SEARCH ENGINE - DO NOT EDIT -->
										<input type="hidden" name="resulttemplate" value="epafiles_default.hts" />
										<input type="hidden" name="Action" value="FilterSearch" />
										<input type="hidden" name="Filter" value="sample2filt.hts" />
										<input type="hidden" name="collection" value="All" />
										<!-- END SEARCH CONTROLS -->
										<!-- START AREA SPECIFIC LINKING -->
										<a href="http://www.epa.gov/storet/whats_new.html">Recent Additions</a> |
										<a href="http://www.epa.gov/storet/contact.html">Contact Us</a> | 

										<!-- END AREA SPECIFIC LINKING -->
										<!-- LINK FOR PRINT VERSION -->
										<!-- CHARACTERS NEED TO APPEAR EXACTLY LIKE THIS - DO NOT EDIT -->
										<a href="http://www.epa.gov/cgi-bin/epaprintonly.cgi">Print 
										Version</a> 
										<!-- END PRINT VERSION -->
										<img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
										width="1" height="1" vspace="2" hspace="5" alt="" /> 
										Search: 
										<input name="queryText" size="10" value="" />
										<input type="image" src="http://www.epa.gov/epafiles/images/epafiles_btn_gosearch.gif" value="Run Search" 
										border="0" width="27" height="16" alt="Run Search" name="image" />
									</td>
								</tr>
								<tr> 
									<td colspan="2"><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
									width="5" height="3" alt="" align="middle" /></td>
								</tr>
								<tr> 
									<!-- BEGIN BREADCRUMBS -->
									<td class="epaHighlightBox"><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
									width="4" height="4" alt="Begin Hierarchical Links" border="0" /></td>
									<td width="100%" class="epaHighlightBox"><span class="epaBreadLinks"><epabreadcrumbs><a href="http://www.epa.gov">EPA 
										Home</a> &gt; 
										<!-- START AREA BREADCRUMBS -->
										<!-- START LINK TO AREA HOME PAGE -->
										<a href="http://www.epa.gov/water/">Water</a>
&gt; <a href="http://www.epa.gov/owow/">Wetlands, Oceans, & Watersheds</a>
&gt; <a href="http://www.epa.gov/owow/monitoring/">Monitoring and Assessing Water Quality</a>
										<!-- END LINK TO AREA HOME PAGE -->
										&gt; <a href="http://www.epa.gov/storet/index.html">STORET</a> 
&gt; ' || p_breadcrumbs || '
										<!-- END AREA BREADCRUMBS -->
										<img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
									width="4" height="4" alt="End Hierarchical Links" 
									border="0" /></epabreadcrumbs></span></td>
									<!-- END BREADCRUMBS -->
								</tr>
							</table>
						</form>
						<!-- END GLOBAL LINKS AND SEARCH -->
					</td>
				</tr>
			</table>
		</td>
	</tr>
</table>
<table width="95%" border="0" cellspacing="0" cellpadding="0">
	<tr> 
		<!--@@@startprint@@@-->
		<td valign="top"><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
		width="5" height="5" alt="" /></td>
		<!--@@@endprint@@@-->
		<td valign="top" width="100%"> 
			<!-- BEGIN HORIZONTAL TASKBAR -->
			<!-- If using a horizontal taskbar, put the code here -->
			<!-- END HORIZONTAL TASKBAR -->
			<a name="pagecontents"></a> 

			<!-- BEGIN PAGE NAME -->
			<h1 class="epaPageName"><!-- #BeginEditable "PageName" --> 
        <!-- If using a right-aligned image next to the page name, insert it here 
			- remember to include the align="right" attribute in the IMG tag -->');

-- S.Kho 8/14/03
/* htp.p('		<!--img alt="STORET logo" src="http://www.epa.gov/storet/images/logo_sm.gif" width="118" height="146" align="left" vspace="6" hspace="12" /-->
        <h3 align="CENTER"><font face="Arial, Helvetica, sans-serif" color="#003399">'||page_title||'</font></h3> ');
*/
htp.p('			<!-- #EndEditable --></h1>
			<!-- END PAGE NAME -->

			<!-- BEGIN CONTENT AREA -->
			<!-- #BeginEditable "content" -->
');

END;


/*********************************************************/    
/* *** STORET Station Web footer  *** */
PROCEDURE WME_footer (
	help_counter number default 9999)
IS
	help_file         varchar2(100);
BEGIN
	-- Generate Help File link
	help_file:=''''|| DW_glob_var.lv_doc_path||'WME_STORET_Help.html#'||help_counter||'''';

	--EPA Standard Footer
htp.print('
      <!-- #EndEditable --> 
			<!-- END CONTENT AREA -->
			<p align="right"><A tabindex="999" HREF="javascript:openHelp('||help_file||')">Help</A></p>
		</td>
	</tr>
</table>
<table width="95%" border="0" cellspacing="0" cellpadding="0">
	<tr> 
		<td valign="top" align="center" rowspan="4"><img src="http://www.epa.gov/epafiles/images/epafiles_misc_space.gif" 
		width="5" height="5" alt="" /></td>
		<td valign="top" width="100%" align="center"> 
			<!-- BEGIN AREA FOOTER -->
			<!-- ADD LINKS AND TEXT FOR AREA FOOTER WITHIN THIS PARAGRAPH TAG -->
			<p class="epaFooterText" align="center"></p>
			<!-- END AREA FOOTER -->
		</td>
	</tr>
	<tr> 
		<td valign="top" width="100%" align="center">&nbsp;</td>
	</tr>
	<tr> 
		<td valign="top" width="100%" align="center"> 
			<!-- BEGIN FOOTER IMAGE -->
			<img src="http://www.epa.gov/epafiles/images/epafiles_misc_dot_dkblue.gif" 
		width="460" height="4" vspace="2" alt="Begin Site Footer" /> 
			<!-- END FOOTER IMAGE -->
		</td>
	</tr>
	<tr> 
		<td valign="top" width="100%" align="center"> 
			<!-- BEGIN GLOBAL FOOTER -->
			<p class="epaFooterText"><a href="http://www.epa.gov/">EPA Home</a> 
				| <a href="http://www.epa.gov/epafiles/usenotice.htm">Privacy 
				and Security Notice</a> | <a href="http://www.epa.gov/storet/contact.html">Contact Us</a></p>
			<p class="epaFooterText"> <epadateurl> 
			<!--<script language="JavaScript" src="http://www.epa.gov/epafiles/scripts/dateurl.js"> -->
			<script language="JavaScript">
document.write("Last updated on December 5, 2003 <br />");
// First, strip off any URL variables
var URLAddress = location.href.substring(0,(location.href.indexOf("?")>0)?location.href.indexOf("?"):location.href.length)

// Remove characters outisde the printable range as well as "<" and ">" which may indicate tags.
var newAddress = '''';
for (i=0;i<URLAddress.length;i++)
{
 if ((URLAddress.charCodeAt(i) > 31 || URLAddress.charCodeAt(i) < 127) && URLAddress.charAt(i) != ''>'' && URLAddress.charAt(i) != ''<'')
  newAddress = newAddress + URLAddress.charAt(i);
}

// write cleaned URL
document.write("URL: " + newAddress);

			</script>
				</epadateurl></p>
			<p>&nbsp;</p>
			<!-- END GLOBAL FOOTER -->
		</td>
	</tr>
</table>
<p>&nbsp;</p>
</body>
</html>
');
END;


/*********************************************************/    
END;
/
SHOW ERRORS;


spool off;

